//
//  ViewController.m
//  SolarSystem
//
//  Created by Philipp Wacker on 22.10.14.
//  Copyright (c) 2014 Media Computing Group. All rights reserved.
//

#import "ViewController.h"
#import <SceneKit/SceneKit.h>
#import <SpriteKit/SpriteKit.h>

@interface ViewController ()

@property (weak, nonatomic) IBOutlet SCNView *sceneView;
//the SceneKit Scene
@property (strong) SCNScene *scene;

//Arrays to store all spheres and all nodes with actions
@property (strong) NSMutableArray *spheres;
@property (strong) NSMutableArray *actionNodes;

//tap counter to decide what action trigger
@property NSInteger tapCounter;

@end

@implementation ViewController

- (void)viewDidLoad {
    [super viewDidLoad];
    self.actionNodes = [[NSMutableArray alloc] init];
    self.spheres = [[NSMutableArray alloc] init];

    //set up basic scene with lights and camera position
    [self setupTheScene];
    
//    // add planets and basic actions
//    [self setupSolarSystem];
    
//    //set up the earth with different material contents
//    [self setupEarth];
    
    //start the physics simulation
    [self setupPhysicsScene];
}

- (void)setupTheScene
{
    self.scene = [[SCNScene alloc] init];
    
    //add a camera node that looks towards the center of the screen depending on the position (disregarding x-position)
    SCNNode *cameraNode = [SCNNode node];
    cameraNode.name = @"CameraNode";
    cameraNode.camera = [SCNCamera camera];
    [self.scene.rootNode addChildNode:cameraNode];
    //right handed coordinate system; y-axis = "height"
    cameraNode.position = SCNVector3Make(0,50,100);
    //rotation around x axis (1,0,0) in radians
    cameraNode.rotation = SCNVector4Make(1, 0, 0, -cameraNode.position.y/sqrt(pow(cameraNode.position.y,2)+pow(cameraNode.position.z, 2)));
    
    //adjust field of view and let the z-range (range where objects can be seen) be adjusted automatically
    cameraNode.camera.xFov = 60;
    cameraNode.camera.automaticallyAdjustsZRange = true;
    
    //place a basic box in the scene
    SCNBox *boxGeometry = [SCNBox boxWithWidth:10 height:10 length:10 chamferRadius:2];
    SCNNode *boxNode = [SCNNode node];
    boxNode.geometry = boxGeometry;
    
    //[self.scene.rootNode addChildNode:boxNode];
    
    //let the view display this scene
    self.sceneView.scene = self.scene;
    
    self.sceneView.autoenablesDefaultLighting = true;
    
    //ambient light to include general lighting
    SCNNode *ambientLightNode = [SCNNode node];
    ambientLightNode.name = @"AmbientLightNode";
    ambientLightNode.light = [SCNLight light];
    ambientLightNode.light.type = SCNLightTypeAmbient;
    ambientLightNode.light.color = [UIColor colorWithWhite:0.67 alpha:1.0];
    [self.scene.rootNode addChildNode:ambientLightNode];
    
    //add light from above
    SCNNode *omniLightNode = [SCNNode node];
    omniLightNode.light = [SCNLight light];
    omniLightNode.light.color = [UIColor colorWithWhite:0.5 alpha:1.0];
    omniLightNode.light.type = SCNLightTypeOmni;
    omniLightNode.position = SCNVector3Make(0, 40, 0);
    [self.scene.rootNode addChildNode:omniLightNode];
    
   // [self setupSolarSystem];

    
}

- (void)setupSolarSystem
{
    //create CenterOfUniverse
    SCNNode *centerOfUniverse = [SCNNode node];
    centerOfUniverse.name = @"CenterOfUniverse";
    [self.scene.rootNode addChildNode:centerOfUniverse];
    
    //create Sun
    SCNNode *sunNode = [SCNNode node];
    sunNode.name = @"SunNode";
    sunNode.geometry = [SCNSphere sphereWithRadius:10];
    //query for specific node by the name; in this case we could also use the centerOfUniverse node directly as we still have a reference
    [[self.scene.rootNode childNodeWithName:@"CenterOfUniverse" recursively:true] addChildNode:sunNode];
    
    //create Earth
    SCNNode *earthNode = [SCNNode node];
    earthNode.name = @"EarthNode";
    earthNode.geometry = [SCNSphere sphereWithRadius:2.5];
    earthNode.position = SCNVector3Make(50, 0, 0);
    [[self.scene.rootNode childNodeWithName:@"CenterOfUniverse" recursively:true] addChildNode:earthNode];
    
    //create Moon
    SCNNode *moonNode = [SCNNode node];
    moonNode.name = @"MoonNode";
    moonNode.geometry = [SCNSphere sphereWithRadius:1];
    moonNode.position = SCNVector3Make(6,0,0);
    [[self.scene.rootNode childNodeWithName:@"EarthNode" recursively:true] addChildNode:moonNode];

//-----------------------------------------------------
    //value to time the rotations (not real values)
    double earthDayDuration = 1;
    
    //rotate earth around the center of the universe
    SCNAction *earthRotation = [SCNAction rotateByAngle:-M_PI aroundAxis:SCNVector3Make(0, 1, 0) duration:earthDayDuration*5];
    earthRotation = [SCNAction repeatActionForever:earthRotation];
    [[self.scene.rootNode childNodeWithName:@"CenterOfUniverse" recursively:true] runAction:earthRotation];
    
    //add center of universe node to actionNodes array to be able to cancel the action later
    [self.actionNodes addObject:centerOfUniverse];
    
    //rotate earth around its own axis (thereby also rotating the moon)
    SCNAction *moonRotation = [SCNAction rotateByAngle:-M_PI aroundAxis:SCNVector3Make(0,1,0) duration:earthDayDuration];
    moonRotation = [SCNAction repeatActionForever:moonRotation];
    [[self.scene.rootNode childNodeWithName:@"EarthNode" recursively:true] runAction:moonRotation];
    
    [self.actionNodes addObject:earthNode];

//-----------------------------------------------------
    //adding textures (more info on materials later)
    SCNMaterial *sunTexture = [SCNMaterial material];
    sunTexture.diffuse.contents = [UIImage imageNamed:@"sun.jpg"];
    sunNode = [self.scene.rootNode childNodeWithName:@"SunNode" recursively:true];
    sunNode.geometry.firstMaterial = sunTexture;
    
    SCNMaterial *earthTexture = [SCNMaterial material];
    earthTexture.diffuse.contents = [UIImage imageNamed:@"earth-diffuse.jpg"];
    earthNode = [self.scene.rootNode childNodeWithName:@"EarthNode" recursively:true];
    earthNode.geometry.firstMaterial = earthTexture;
    
    SCNMaterial *moonTexture = [SCNMaterial material];
    moonTexture.diffuse.contents = [UIImage imageNamed:@"moon.jpg"];
    moonNode = [self.scene.rootNode childNodeWithName:@"MoonNode" recursively:true];
    moonNode.geometry.firstMaterial = moonTexture;

//-----------------------------------------------------
    //let sun rotate in other direction
    SCNAction *sunRotation = [SCNAction rotateByAngle:M_PI aroundAxis:SCNVector3Make(0, 1, 0) duration:4];
    sunRotation = [SCNAction repeatActionForever:sunRotation];
    [sunNode runAction:sunRotation];
    
    [self.actionNodes addObject:sunNode];
    
////-----------------------------------------------------
//    //use SpriteKit scene as texture
//    SKScene *spriteKitTexture = [[SKScene alloc] initWithSize:CGSizeMake(1000, 1000)];
//    //Background
//    SKSpriteNode *background = [[SKSpriteNode alloc] initWithImageNamed:@"space.jpg"];
//    background.position = CGPointMake(500, 500);
//    background.size = CGSizeMake(1000, 1000);
//    [spriteKitTexture addChild: background];
//    //Spaceship
//    SKSpriteNode *spaceship = [[SKSpriteNode alloc] initWithImageNamed:@"Spaceship.png"];
//    spaceship.position = CGPointMake(200, 500);
//    spaceship.zRotation = 1.5 * M_PI;
//    SKAction *leftToRight = [SKAction moveBy:CGVectorMake(400, 0) duration:2.0];
//    SKAction *turnAround = [SKAction rotateByAngle:M_PI duration:2.0];
//    SKAction *rightToLeft = [SKAction moveBy:CGVectorMake(-400, 0) duration:2.0];
//    //specify action sequence for the spaceship
//    SKAction *spaceshipAction = [SKAction repeatActionForever:[SKAction sequence:@[leftToRight, turnAround, rightToLeft, turnAround]]];
//    [spaceship runAction:spaceshipAction];
//
//    [spriteKitTexture addChild:spaceship];
//    sunTexture.diffuse.contents = spriteKitTexture;

////-----------------------------------------------------
    //let camera follow earth
    [self.scene.rootNode childNodeWithName:@"CameraNode" recursively:true].constraints = @[[SCNLookAtConstraint lookAtConstraintWithTarget:earthNode]];

    //store all spheres in our array
    [self.spheres addObjectsFromArray:@[sunNode,earthNode,moonNode]];
}

- (void)setupEarth
{
    //setup basic earth with diffuse texture
    SCNNode *earthNode = [SCNNode node];
    earthNode.name = @"EarthNode";
    earthNode.geometry = [SCNSphere sphereWithRadius:25];
    [self.scene.rootNode addChildNode:earthNode];
    
    SCNMaterial *earthMaterial = [SCNMaterial material];
    earthMaterial.diffuse.contents = [UIImage imageNamed:@"earth-diffuse.jpg"];
    earthNode = [self.scene.rootNode childNodeWithName:@"EarthNode" recursively:true];
    earthNode.geometry.firstMaterial = earthMaterial;
    
    SCNAction *earthRotation = [SCNAction rotateByAngle:-M_PI aroundAxis:SCNVector3Make(0, 1, 0) duration:15];
    earthRotation = [SCNAction repeatActionForever:earthRotation];
    [earthNode runAction:earthRotation];

//-----------------------------------------------------
    //disable automatic lighting (is also automatically done when you add an own light source)
    self.sceneView.autoenablesDefaultLighting = NO;
    //add rotating omni light
    SCNNode *rotatingLightAnchor = [SCNNode node];
    [self.scene.rootNode addChildNode:rotatingLightAnchor];
    
    SCNNode *sceneLightNode = [SCNNode node];
    sceneLightNode.light = [SCNLight light];
    sceneLightNode.light.color = [UIColor colorWithWhite:0.9 alpha:1.0];
    sceneLightNode.light.type = SCNLightTypeOmni;
    sceneLightNode.position = SCNVector3Make(0, 50, 100);
    
    [rotatingLightAnchor runAction:[SCNAction repeatActionForever:[SCNAction rotateByAngle:M_PI aroundAxis:SCNVector3Make(0, 1, 0) duration:10]]];
    
    [rotatingLightAnchor addChildNode:sceneLightNode];
    
//-----------------------------------------------------
    //add ambient light
    SCNNode *ambientLightNode = [SCNNode node];
    ambientLightNode.name = @"AmbientLightNode";
    ambientLightNode.light = [SCNLight light];
    ambientLightNode.light.type = SCNLightTypeAmbient;
    ambientLightNode.light.color = [UIColor colorWithWhite:0.2 alpha:1.0];
    [self.scene.rootNode addChildNode:ambientLightNode];

//-----------------------------------------------------
//    //define specular reflection of the earth
//    //solid color
    earthMaterial.specular.contents = [UIColor whiteColor];
    earthMaterial.shininess = 0.1;
//    //with map to only reflect on ocean
    earthMaterial.specular.contents = [UIImage imageNamed:@"earth-specular.jpg"];
 
////-----------------------------------------------------
    //add a normal map
    //from file
  //  earthMaterial.normal.contents = [UIImage imageNamed:@"earth-bump.png"];
//    //generating normal map
    earthMaterial.normal.contents = [[SKTexture textureWithImageNamed:@"earth-diffuse.jpg"] textureByGeneratingNormalMap];

//-----------------------------------------------------
//    //set a reflective map (environment)
    earthMaterial.reflective.contents = [UIImage imageNamed:@"earth-reflective.jpg"];
    earthMaterial.specular.intensity = 0;
    
//-----------------------------------------------------
//    //set emission
    earthMaterial.emission.contents = [UIImage imageNamed:@"earth-emissive.jpg"];
    earthMaterial.emission.intensity = 0.5;
//    //turn off ambient light
    [self.scene.rootNode childNodeWithName:@"AmbientLightNode" recursively:true].light.color = [UIColor blackColor];
    
//-----------------------------------------------------
//    //add clouds
//    //sphere
    SCNNode *cloudNode = [SCNNode node];
    cloudNode.name = @"CloudNode";
    cloudNode.geometry = [SCNSphere sphereWithRadius:27];
    [earthNode addChildNode:cloudNode];
    cloudNode.opacity = 0.8;
    //transparency
    cloudNode.geometry.firstMaterial.diffuse.contents = [UIImage imageNamed:@"cloudsTransparency.png"];
    cloudNode.geometry.firstMaterial.transparent.contents = [UIImage imageNamed:@"cloudsTransparency.png"];
    cloudNode.geometry.firstMaterial.transparencyMode = SCNTransparencyModeRGBZero;
}

- (void)setupPhysicsScene
{
    self.tapCounter = 0;
    
    SCNNode *floorNode = [SCNNode node];
    
    SCNFloor *floorGeometry = [[SCNFloor alloc] init];
    //floorGeometry.reflectionFalloffStart = 1.0;
    //floorGeometry.reflectionFalloffEnd = 0;
    //floorGeometry.reflectivity = 0;
    floorNode.geometry = floorGeometry;
    UIImage *floorTexture = [UIImage imageNamed:@"floor-frames.jpg"];
    floorNode.geometry.firstMaterial.diffuse.contents = floorTexture;
    floorNode.geometry.firstMaterial.diffuse.wrapS = SCNWrapModeRepeat;
    floorNode.geometry.firstMaterial.diffuse.wrapT = SCNWrapModeRepeat;
    
    floorNode.position = SCNVector3Make(0, -15, 0);
    
    //add physics body
    floorNode.physicsBody = [SCNPhysicsBody bodyWithType:SCNPhysicsBodyTypeStatic shape:[SCNPhysicsShape shapeWithGeometry:floorNode.geometry options:nil]];
    
    [self.scene.rootNode addChildNode:floorNode];
    
}

- (IBAction)touchReceived:(id)sender
{
    if (self.tapCounter == 0)
    {
        //stop all actions
        for (SCNNode *currentNode in self.actionNodes)
        {
            [currentNode removeAllActions];
        }
        //set physic bodies on all spheres
        for (SCNNode *currentNode in self.spheres)
        {
            currentNode.physicsBody = [SCNPhysicsBody bodyWithType:SCNPhysicsBodyTypeDynamic shape:[SCNPhysicsShape shapeWithGeometry:currentNode.geometry options:nil]];
            //give the physics bodies different masses (react differently to forces etc)
            if ([currentNode.name isEqualToString:@"SunNode"])
            {
                currentNode.physicsBody.mass = 10;
            }
            else if ([currentNode.name isEqualToString:@"EarthNode"])
            {
                currentNode.physicsBody.mass = 5;
            }
            else
            {
                currentNode.physicsBody.mass = 2;
            }
        }
        self.tapCounter++;
    }
    else if (self.tapCounter == 1)
    {
        //add 200 small spheres and drop them in the scene
        for (int i = 0; i < 200; i++)
        {
            SCNNode *newNode = [SCNNode node];
            newNode.geometry = [SCNSphere sphereWithRadius:1];
            newNode.physicsBody = [SCNPhysicsBody bodyWithType:SCNPhysicsBodyTypeDynamic shape:[SCNPhysicsShape shapeWithGeometry:newNode.geometry options:nil]];
            //define "bounciness"
            newNode.physicsBody.restitution = 0.8;
            newNode.physicsBody.mass = 0.5;
            
            //random positioning
            newNode.position = SCNVector3Make(-40 + rand()%80, 25 + rand()%10, -40+ rand()%80);
            
            [self.scene.rootNode addChildNode:newNode];
            
            [self.spheres addObject:newNode];
            
        }
        self.tapCounter++;
    }
    else
    {
        //apply a force to all spheres to let them fly away
        for (SCNNode *currentNode in self.spheres)
        {
            //vector from the center to the center of the sphere
            SCNVector3 direction = currentNode.position;
            direction.y += rand()%100;
            [currentNode.physicsBody applyForce:direction atPosition:SCNVector3Make(0, 0, 0) impulse:YES];
        }
    }
}


- (void)didReceiveMemoryWarning {
    [super didReceiveMemoryWarning];
    // Dispose of any resources that can be recreated.
}

@end
